<?php

namespace App\Http\Controllers\Admin;

use App\Http\Services\TransferServices;
use App\Http\Controllers\Controller;
use App\Http\Services\PaymentServices;
use App\Http\Services\WalletServices;
use App\Models\BankWithdrawal;
use App\Models\GeneralSetting;
use App\Models\Transaction;
use App\Models\User;
use App\Models\WithdrawMethod;
use App\Models\Withdrawal;
use Carbon\Carbon;
use Illuminate\Http\Request;

class WithdrawalController extends Controller
{
    public function pending()
    {
        $pageTitle = 'Pending Withdrawals';
        $withdrawals = Withdrawal::pending()->with(['user','method'])->orderBy('id','desc')->paginate(getPaginate());
        $emptyMessage = 'No withdrawal found';
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'withdrawals', 'emptyMessage'));
    }

    public function approved()
    {
        $pageTitle = 'Approved Withdrawals';
        $withdrawals = Withdrawal::approved()->with(['user','method'])->orderBy('id','desc')->paginate(getPaginate());
        $emptyMessage = 'No withdrawal found';
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'withdrawals', 'emptyMessage'));
    }

    public function rejected()
    {
        $pageTitle = 'Rejected Withdrawals';
        $withdrawals = Withdrawal::rejected()->with(['user','method'])->orderBy('id','desc')->paginate(getPaginate());
        $emptyMessage = 'No withdrawal found';
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'withdrawals', 'emptyMessage'));
    }

    public function log()
    {
        $pageTitle = 'Withdrawals Log';
        $withdrawals = Withdrawal::where('status', '!=', 0)->with(['user','method'])->orderBy('id','desc')->paginate(getPaginate());
        $emptyMessage = 'No withdrawal history';
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'withdrawals', 'emptyMessage'));
    }


    public function logViaMethod($methodId,$type = null){
        $method = WithdrawMethod::findOrFail($methodId);
        if ($type == 'approved') {
            $pageTitle = 'Approved Withdrawal Via '.$method->name;
            $withdrawals = Withdrawal::where('status', 1)->with(['user','method'])->where('method_id',$method->id)->orderBy('id','desc')->paginate(getPaginate());
        }elseif($type == 'rejected'){
            $pageTitle = 'Rejected Withdrawals Via '.$method->name;
            $withdrawals = Withdrawal::where('status', 3)->with(['user','method'])->where('method_id',$method->id)->orderBy('id','desc')->paginate(getPaginate());

        }elseif($type == 'pending'){
            $pageTitle = 'Pending Withdrawals Via '.$method->name;
            $withdrawals = Withdrawal::where('status', 2)->with(['user','method'])->where('method_id',$method->id)->orderBy('id','desc')->paginate(getPaginate());
        }else{
            $pageTitle = 'Withdrawals Via '.$method->name;
            $withdrawals = Withdrawal::where('status', '!=', 0)->with(['user','method'])->where('method_id',$method->id)->orderBy('id','desc')->paginate(getPaginate());
        }
        $emptyMessage = 'No withdrawal found';
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'withdrawals', 'emptyMessage','method'));
    }


    public function search(Request $request, $scope)
    {
        $search = $request->search;
        $emptyMessage = 'No search result found.';

        $withdrawals = Withdrawal::with(['user', 'method'])->where('status','!=',0)->where(function ($q) use ($search) {
            $q->where('trx', 'like',"%$search%")
                ->orWhereHas('user', function ($user) use ($search) {
                    $user->where('username', 'like',"%$search%");
                });
        });

        if ($scope == 'pending') {
            $pageTitle = 'Pending Withdrawal Search';
            $withdrawals = $withdrawals->where('status', 2);
        }elseif($scope == 'approved'){
            $pageTitle = 'Approved Withdrawal Search';
            $withdrawals = $withdrawals->where('status', 1);
        }elseif($scope == 'rejected'){
            $pageTitle = 'Rejected Withdrawal Search';
            $withdrawals = $withdrawals->where('status', 3);
        }else{
            $pageTitle = 'Withdrawal History Search';
        }

        $withdrawals = $withdrawals->paginate(getPaginate());
        $pageTitle .= ' - ' . $search;

        return view('admin.withdraw.withdrawals', compact('pageTitle', 'emptyMessage', 'search', 'scope', 'withdrawals'));
    }

    public function dateSearch(Request $request,$scope){
        $search = $request->date;
        if (!$search) {
            return back();
        }
        $date = explode('-',$search);
        $start = @$date[0];
        $end = @$date[1];

        // date validation
        $pattern = "/\d{2}\/\d{2}\/\d{4}/";
        if ($start && !preg_match($pattern,$start)) {
            $notify[] = ['error','Invalid date format'];
            return redirect()->route('admin.withdraw.log')->withNotify($notify);
        }
        if ($end && !preg_match($pattern,$end)) {
            $notify[] = ['error','Invalid date format'];
            return redirect()->route('admin.withdraw.log')->withNotify($notify);
        }


        if ($start) {
            $withdrawals = Withdrawal::where('status','!=',0)->whereDate('created_at',Carbon::parse($start));
        }
        if($end){
            $withdrawals = Withdrawal::where('status','!=',0)->whereDate('created_at','>=',Carbon::parse($start))->whereDate('created_at','<=',Carbon::parse($end));
        }
        if ($request->method) {
            $method = WithdrawMethod::findOrFail($request->method);
            $withdrawals = $withdrawals->where('method_id',$method->id);
        }

        if ($scope == 'pending') {
            $withdrawals = $withdrawals->where('status', 2);
        }elseif($scope == 'approved'){
            $withdrawals = $withdrawals->where('status', 1);
        }elseif($scope == 'rejected') {
            $withdrawals = $withdrawals->where('status', 3);
        }

        $withdrawals = $withdrawals->with(['user', 'method'])->paginate(getPaginate());
        $pageTitle = 'Withdraw Log';
        $emptyMessage = 'No Withdrawals Found';
        $dateSearch = $search;
        return view('admin.withdraw.withdrawals', compact('pageTitle', 'emptyMessage', 'dateSearch', 'withdrawals','scope'));


    }

    public static function addUpdateTransaction(
        string $type,
        string $desc = null,
        string $reference,
        int $status = 1
    ) {
        try {

            $withdraw = Withdrawal::where('trx', $reference)->first();

            $user = User::find($withdraw->user_id);

            // Find Transaction
            $transaction = Transaction::where('trx',$withdraw->trx)->first();

            // Verify transaction
            if($transaction) {
                //
                if($desc) $transaction->details = $desc;
                $transaction->status = $status;
                $transaction->save();

                return $transaction;
            }

            // Add Transaction
            $transaction = new Transaction();
            $transaction->user_id = $withdraw->user_id;
            $transaction->type =  $type;
            $transaction->amount = $withdraw->amount;
            if($withdraw->is_bonus) {
                $transaction->post_balance = $user->bonus_balance;
            }else{
                $transaction->post_balance = $user->balance;
            }
            $transaction->charge = $withdraw->charge;
            $transaction->currency = $withdraw->currency;
            $transaction->trx_type = '-';
            $transaction->details = $desc;
            $transaction->trx =  $withdraw->trx;
            $transaction->status =  $status;
            $transaction->save();

            return $transaction;
        } catch (\Exception $th) {
            //throw $th;
            return $th;
        }
    }

    public function details($id)
    {
        $withdrawal = Withdrawal::where('id',$id)->where('status', '!=', 0)->with(['user','method'])->firstOrFail();
        $pageTitle = $withdrawal->user->username.' Withdraw Requested ' . showAmount($withdrawal->amount) . ' '.config('settings')->cur_text;
        $details = $withdrawal->withdraw_information ? json_encode($withdrawal->withdraw_information) : null;

        $methodImage =  getImage(imagePath()['withdraw']['method']['path'].'/'. $withdrawal->method->image,'800x800');

        return view('admin.withdraw.detail', compact('pageTitle', 'withdrawal','details','methodImage'));
    }

    function approve_all( Request $request ){
        $ids = $request->ids;

        $k = 0;
		$idList = ($ids) ? $ids : [];

        foreach ($idList as $id){

            $withdraw = Withdrawal::where('id',$id)->where('status',2)->first();

            if($withdraw) {

                $withdraw->status = 1;
                $withdraw->admin_feedback = 'Payment made and approved';

                if($request->is_manual){
                    $withdraw->save();
                }else{
                    $confirm = new TransferServices();
                    $confirm->initiate_transfer($request->id);
                }

                // Update Transaction
                self::addUpdateTransaction('withdraw', null, $withdraw->trx, 1);

                notify($withdraw->user, 'WITHDRAW_APPROVE', [
                    'method_name' => $withdraw->method->name,
                    'method_currency' => $withdraw->currency,
                    'method_amount' => showAmount($withdraw->final_amount),
                    'amount' => showAmount($withdraw->amount),
                    'charge' => showAmount($withdraw->charge),
                    'currency' => config('settings')->cur_text,
                    'rate' => showAmount($withdraw->rate),
                    'trx' => $withdraw->trx,
                    'admin_details' => $request->details
                ]);
                $k++;
            }


        }


        $notify[] = ['success', "$k Withdrawal marked as approved."];
        return redirect()->route('admin.withdraw.pending')->withNotify($notify);
    }

    public function approve(Request $request)
    {
        $request->validate(['id' => 'required|integer']);
        $withdraw = Withdrawal::where('id',$request->id)->where('status',2)->with('user')->firstOrFail();
        $withdraw->status = 1;
        $withdraw->admin_feedback = $request->details;


        if($request->is_manual){
            $withdraw->save();
        }else{
            $confirm = new TransferServices();
            $confirm->initiate_transfer($request->id);
        }


        // Update Transaction
        self::addUpdateTransaction('withdraw', null, $withdraw->trx, 1);


        if($withdraw->is_bonus){
            $msg = "Congratulations ".$withdraw->user->username.", your referral withdrawal of $withdraw->currency $withdraw->amount has been successfully sent to your local bank account, keep referring Keep earning";
        }else {
            $msg = "Congratulations ".$withdraw->user->username.", you have successfully sold your dollar to us and the sum of $withdraw->currency $withdraw->amount has been successfully sent to your bank account";
        }

        $msg = "Congratulations ".$withdraw->user->username.", your withdrawal of $withdraw->currency ".number_format($withdraw->amount)." has been successfully sent to your local bank account";
        send_to_telegram($msg);


        notify($withdraw->user, 'WITHDRAW_APPROVE', [
            'method_name' => $withdraw->method->name,
            'method_currency' => $withdraw->currency,
            'method_amount' => showAmount($withdraw->final_amount),
            'amount' => showAmount($withdraw->amount),
            'charge' => showAmount($withdraw->charge),
            'currency' => config('settings')->cur_text,
            'rate' => showAmount($withdraw->rate),
            'trx' => $withdraw->trx,
            'admin_details' => $request->details
        ]);

        $notify[] = ['success', 'Withdrawal marked as approved.'];
        return redirect()->route('admin.withdraw.pending')->withNotify($notify);
    }

    public function autopay(Request $request, PaymentServices $payment, WalletServices $wallet)
    {
        $request->validate(['id' => 'required|integer']);
        $withdraw = Withdrawal::where('id',$request->id)->where('status',2)->with('user')->firstOrFail();

        $user = User::find($withdraw->user_id);
        $general = GeneralSetting::first();

        $status_text = 'Withdrawal marked as approved.';
        $status = 1;

        // Find Client Bank Account
        $bankData = BankWithdrawal::where('withdrawal_id', $request->id)->firstOrFail();

        // Transfer Payment
        $transferPayment = $payment->payout($withdraw->trx, 'NGN', $withdraw->final_amount, $general->auto_transfer_default, $bankData->bank_code, $bankData->account_number, $bankData->account_name);

        // Exception
        if($transferPayment['status'] == false) {

            // Update User Balance
            $wallet->creditAvailableBalance($user, $withdraw->charge_type, $withdraw->currency, $withdraw->amount);

            // Update Transaction
            self::addUpdateTransaction('withdraw', null, $withdraw->trx, 3);

            $status_text = $transferPayment['message'];
            $status = 3;
        }

        $withdraw->status = $status;
        $withdraw->save();

        /*if($withdraw->is_bonus){
            $msg = "Congratulations ".$withdraw->user->username.", your referral withdrawal of $withdraw->currency $withdraw->amount has been successfully sent to your local bank account, keep referring Keep earning";
        }else {
            $msg = "Congratulations ".$withdraw->user->username.", you have successfully sold your dollar to us and the sum of $withdraw->currency $withdraw->amount has been successfully sent to your bank account";
        }

        $msg = "Congratulations ".$withdraw->user->username.", your withdrawal of $withdraw->currency ".number_format($withdraw->amount)." has been successfully sent to your local bank account";
        send_to_telegram($msg);*/

        $notify[] = ['success', $status_text];
        return redirect()->route('admin.withdraw.pending')->withNotify($notify);
    }

    function reject_all(
        WalletServices $wallet,
        Request $request
        ){
        $ids = $request->ids;

        $k = 0;

        foreach ($ids as $id){

            $withdraw = Withdrawal::where('id',$id)->where('status',2)->first();

            if($withdraw) {

                $withdraw->status = 3;
                $withdraw->admin_feedback = $request->details;
                $withdraw->save();

                $user = User::find($withdraw->user_id);

                // Update User Balance
                $wallet->creditAvailableBalance($user, $withdraw->charge_type, $withdraw->currency, $withdraw->amount);

                // Update Transaction
                self::addUpdateTransaction('withdraw', showAmount($withdraw->amount) . ' ' . config('settings')->cur_text . ' Refunded from withdrawal rejection', $withdraw->trx, 3);

                notify($user, 'WITHDRAW_REJECT', [
                    'method_name' => $withdraw->method->name,
                    'method_currency' => $withdraw->currency,
                    'method_amount' => showAmount($withdraw->final_amount),
                    'amount' => showAmount($withdraw->amount),
                    'charge' => showAmount($withdraw->charge),
                    'currency' => config('settings')->cur_text,
                    'rate' => showAmount($withdraw->rate),
                    'trx' => $withdraw->trx,
                    'post_balance' => showAmount($user->balance),
                    'admin_details' => $request->details
                ]);
                $k++;
            }


        }


        $notify[] = ['success', "$k Withdrawal has been rejected."];
        return redirect()->route('admin.withdraw.pending')->withNotify($notify);
    }


    public function reject(
        WalletServices $wallet,
        Request $request
        )
    {
        $request->validate(['id' => 'required|integer']);
        $withdraw = Withdrawal::where('id',$request->id)->where('status',2)->firstOrFail();

        $withdraw->status = 3;
        $withdraw->admin_feedback = $request->details;
        $withdraw->save();

        $user = User::find($withdraw->user_id);

        // Update User Balance
        $wallet->creditAvailableBalance($user, $withdraw->charge_type, $withdraw->currency, $withdraw->amount);


		/*$transaction = new Transaction();
		$transaction->user_id = $withdraw->user_id;
		$transaction->amount = $withdraw->amount;
		$transaction->post_balance = $user->balance;
		$transaction->charge = 0;
		$transaction->trx_type = '+';
		$transaction->details = showAmount($withdraw->amount) . ' ' . config('settings')->cur_text . ' Refunded from withdrawal rejection';
		$transaction->trx = $withdraw->trx;
		$transaction->save();*/
		// Update Transaction
		self::addUpdateTransaction('withdraw', showAmount($withdraw->amount) . ' ' . config('settings')->cur_text . ' Refunded from withdrawal rejection', $withdraw->trx, 3);


        notify($user, 'WITHDRAW_REJECT', [
            'method_name' => $withdraw->method->name,
            'method_currency' => $withdraw->currency,
            'method_amount' => showAmount($withdraw->final_amount),
            'amount' => showAmount($withdraw->amount),
            'charge' => showAmount($withdraw->charge),
            'currency' => config('settings')->cur_text,
            'rate' => showAmount($withdraw->rate),
            'trx' => $withdraw->trx,
            'post_balance' => showAmount($user->balance),
            'admin_details' => $request->details
        ]);

        $notify[] = ['success', 'Withdrawal has been rejected.'];
        return redirect()->route('admin.withdraw.pending')->withNotify($notify);
    }

}
