<?php

namespace App\Http\Controllers\Gateway;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Withdrawal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class IpnWKPayController extends Controller
{
    public function ipnDeposit(Request $request)
    {

        try {

            // Validate request
            $validator = Validator::make($request->all(), [
                'order_no' => 'required|string',
                'sys_no' => 'required|string',
                'order_amount' => 'required',
                'status' => 'required|in:success,1'
            ]);

            // Validation Failed
            if ($validator->fails()) throw new \Exception(implode('...', $validator->errors()->all()));

            // Find Transaction
            $recharge = Deposit::where('trx', $request->order_no)->first();

            if(!$recharge) {
                $notify = ['success' => true, 'status' => 'ok', 'message' => ['success' => 'Transaction reference not found']];
                return response($notify, 200);
            }

            // Deposit Handle
            $updateData = self::updateDeposit($recharge, $request->order_amount, $request->all());

            // Exception
            if($updateData instanceof \Exception) throw new \Exception($updateData->getMessage());

            /*$notify = ['success' => true, 'status' => 'ok', 'message'=>['success'=> 'Transaction was successful']];
            return response($notify, 200);*/
            return 'success';
        } catch (\Exception $th) {
            //throw $th;
            $notify = ['success' => true, 'status' => 'error', 'message'=>['error'=> $th->getMessage()]];
            return response($notify, 400);
        }

    }

    /**
     * Deposit update transaction
     */
    public static function updateDeposit( Deposit $deposit, string $amount, array $data ) {
        try {

            // Verify is transaction exisit
            if (!$deposit) throw new \Exception('Transaction not found');

            // Verify if transaction status not complete
            if (in_array($deposit->status, [1, 3])) throw new \Exception('Transaction was already complete');

            // Update transaction details
            $deposit->detail = $data;
            $deposit->status = 1;
            $deposit->save();

			$uBalance = 'balance';
			if(config('settings')->cur_text != 'NGN'){
				$uBalance = 'usdt_balance';
			}

            $user = User::find($deposit->user_id);

            // Update user balance
            $user->$uBalance = $user->$uBalance + $deposit->amount;
            $user->save();

            $transaction = Transaction::where('trx', $deposit->trx)->first();
            $transaction->status = 1;
            $transaction->save();

            return $deposit;
        } catch (\Exception $th) {
            return $th;
        }
    }

    public function ipnTransfer(Request $request)
    {

        try {

            // Validate request
            $validator = Validator::make($request->all(), [
                'order_no' => 'required|string',
                'order_amount' => 'required',
                'result' => 'required|in:success,fail,1'
            ]);

            // Validation Failed
            if ($validator->fails()) throw new \Exception(implode('...', $validator->errors()->all()));

            // Find Transaction
            $payout = Withdrawal::where('trx', $request->order_no)->first();

            if(!$payout) {
                $notify = ['success' => true, 'status' => 'ok', 'message' => ['success' => 'Transaction reference not found']];
                return response($notify, 200);
            }

            // Payout Handle
            $updateData = self::updatePayout($request->result, $payout, $request->order_amount, $request->all());

            // Exception
            if($updateData instanceof \Exception) throw new \Exception($updateData->getMessage());

            /*$notify = ['success' => true, 'status' => 'ok', 'message'=>['success'=> 'Transaction was successful']];
            return response($notify, 200);*/
            return 'success';
        } catch (\Exception $th) {
            //throw $th;
            $notify = ['success' => true, 'status' => 'error', 'message'=>['error'=> $th->getMessage()]];
            return response($notify, 400);
        }

    }

    /**
     * Payout update transaction
     */
    public static function updatePayout( string $status, Withdrawal $payout, string $amount, array $data)
	{
        try {
            // Verify is payout exisit
            if (!$payout) throw new \Exception('Transaction not found');

            $user = User::find($payout->user_id);

            // Verify if payout status not complete
            if (in_array($payout->status, [3])) throw new \Exception('Transaction was already complete');

            // Verify successful status
            if(in_array($status, ['success'])) {
                // Update payout
                $payout->status = 1;
                $payout->save();

                $transaction = Transaction::where('trx', $payout->trx)->first();
                $transaction->status = 1;
                $transaction->save();
            }

            // Verify failed status | Refund amount to user
            if(in_array($status, ['fail'])) {
                // Update payout
                $payout->status = 3;
                $payout->save();

                // Refund amount to User Wallet
                $user->income = $user->income + $payout->amount;
                $user->save();

                $transaction = Transaction::where('trx', $payout->trx)->first();
                $transaction->status = 3;
                $transaction->save();
            }

            return $payout;
        } catch (\Exception $th) {
            return $th;
        }
    }
}
