<?php

namespace App\Http\Controllers;

use App\Http\Component\AppController;
use App\Http\Services\PaymentServices;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Models\Notification;
use App\Models\Bank;
use App\Models\BankAccount;
use App\Models\BankWithdrawal;
use App\Models\Board;
use App\Models\DailyCheckIn;
use App\Models\DailyTask;
use App\Models\Deposit;
use App\Models\Frontend;
use App\Models\GeneralSetting;
use App\Models\Investment;
use App\Models\Plan;
use App\Models\Task;
use App\Models\Transaction;
use App\Models\User;
use App\Models\WithdrawMethod;
use App\Models\Withdrawal;
use App\Rules\FileTypeValidate;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use App\Http\Services\WalletServices;
use App\Models\Blog;
use App\Models\Team;

class UserController extends AppController
{
    private $wallet;
    private $activeTemplate;

    public function __construct(
        WalletServices $wallet
    )
    {
        $this->activeTemplate = activeTemplate();
        $this->wallet = $wallet;
    }

    function daily_check_in(Request $request){

        $user = \auth()->user();
        $today = date("Y-m-d 00:00:00");
        $logins = DailyCheckIn::where('user_id', $user->id)
            ->where('created_at','>=', $today)
            ->count();

		$invests = Investment::where('user_id', $user->id)->count();

		if($invests < 1){
			$notify[] = ['error', 'Kindly invest before claim your bonus'];
            return $this->handleResponse($request, 'Kindly invest before claim your bonus', 401, $notify);
		}
		
		$bType = ($request->bonus)? 2 : 1;
		$wType = ($request->bonus)? 3 : 1;

        if($logins == 0){
            give_bonus($user->id, $bType, config('settings')->cur_text, config('settings')->reward_login, "Daily Check In", $wType);

            $check = new DailyCheckIn();
            $check->user_id = $user->id;
            $check->save();

            $notify[] = ['success', 'Daily check-in bonus claimed successfully.'];
            return $this->handleResponse($request, 'Daily check-in bonus claimed successfully.', 200, $notify);
        }else{
            $notify[] = ['error', 'You have previously claimed daily check-in bonus before.'];
            return $this->handleResponse($request, 'You have previously claimed daily check-in bonus before.', 200, $notify);
        }
    }

    function about(){
        $pageTitle = "About Us";

        return view($this->activeTemplate. 'user.about', compact(
            'pageTitle'
        ));
    }

    function company(){
        $pageTitle = 'About us';
        return view($this->activeTemplate.'user.company',compact('pageTitle'));
    }

    function ceoIntro(){
        $pageTitle = "About Us";

        return view($this->activeTemplate. 'user.ceo_intro', compact(
            'pageTitle'
        ));
    }

    function faqs(){
        $pageTitle = "Frequently Asked Questions";

        return view($this->activeTemplate. 'user.faqs', compact(
            'pageTitle'
        ));
    }

    function appHelp(){
        $pageTitle = "System help";

        return view($this->activeTemplate. 'user.help', compact(
            'pageTitle'
        ));
    }

    function onlineSupport(){
        $pageTitle = "Online Support";

        return view($this->activeTemplate. 'user.online_support', compact(
            'pageTitle'
        ));
    }

    public function home(Request $request)
    {
        $pageTitle = 'Dashboard';
        $user = Auth::user();
		
        $userBalance = $this->wallet->getBalanceRaw($user, config('settings')->cur_text);
		
        if($userBalance instanceOf \Exception) {
            $notify[] = ['error', $userBalance->getMessage()];
            return back()->withNotify($notify);
        }

        $plans = Plan::where('platform', config('settings')->active_template)->where('status', 1)->paginate(getPaginate());
        $today = DailyCheckIn::where('user_id', $user->id)->where('created_at','>=', date("Y-m-d"))->count();

        if($request->claim == 1){
            if($today == 0){
                $desc = "Daily Bonus";
				give_bonus($user->id, 1, config('settings')->cur_text, config('settings')->reward_login, $desc, 2);

                $in = new DailyCheckIn();
                $in->user_id = $user->id;
                $in->save();

            }

            $link = route('invite', ['code' => auth()->user()->ref_code]);

            $notice[] = ['success', "Claimed successfully"];
            return redirect()->away('https://api.whatsapp.com/send?text=Claim bonuses and withdraw to your bank account with Equinixstocks. Use my invite link '. $link);
        }

        $blogs = [];
		$pages = getContent(config('settings')->active_template, 'blogs.element', false, 3, false, 34);

        $emptyMessage = 'Sorry no any thing to show';
        return view($this->activeTemplate. 'user.home.view', compact(
            'pageTitle',
            'user',
            'emptyMessage',
            'userBalance',
            'today',
            'plans',
            'blogs',
            'pages'
        ));
    }

    public function pageBlog($id)
    {
        try {

            // Verify id
			$blog = Frontend::where('slug', $id)->first();
			
            // Exception
            if(!$blog) throw new \Exception("Sorry news not available");
            
			// dd($blog->data_values);
            $pageTitle = $blog->data_values->title;
            $user = Auth::user();
            
            $emptyMessage = 'Data Not Found';
            return view($this->activeTemplate. 'blog.update', compact(
                'pageTitle',
                'user',
                'blog'
            ));
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }

    public function blogPage($id)
    {
        try {

            // Verify id
            $blog = Blog::where(['slug' => $id, 'status' => 1])->first();

            // Exception
            if(!$blog) throw new \Exception("Sorry news not available");
            
            $pageTitle = $blog->name;
            $user = Auth::user();
            
            $emptyMessage = 'Data Not Found';
            return view($this->activeTemplate. 'blog.page', compact(
                'pageTitle',
                'user',
                'blog'
            ));
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }

    public function blog()
    {
        try {

            // Get all blogs
            $blogs = Blog::where(['status' => 1])->get();
			$pages = getContent(config('settings')->active_template, 'blogs.element');
            
            $pageTitle = 'News';
            $user = Auth::user();
            
            $emptyMessage = 'Sorry no any news to show';
            return view($this->activeTemplate. 'blog.all', compact(
                'pageTitle',
                'user',
                'pages',
                'blogs',
                'emptyMessage'
            ));
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }

    public function taskPage($id)
    {
        try {

            // Verify id
            $task = Task::where(['slug' => $id, 'status' => 1])->first();

            // Exception
            if(!$task) throw new \Exception("Sorry task not available");
            
            $pageTitle = $task->name;
            $user = Auth::user();
            
            $emptyMessage = 'Data Not Found';
            return view($this->activeTemplate. 'task.page', compact(
                'pageTitle',
                'user',
                'task'
            ));
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }

    public function task()
    {
        try {

            // Get all tasks
            $tasks = Task::where(['status' => 1])->get();
            
            $pageTitle = 'Tasks';
            $user = Auth::user();
            
            $emptyMessage = 'Sorry no any tasks to show';
            return view($this->activeTemplate. 'task.all', compact(
                'pageTitle',
                'user',
                'tasks',
                'emptyMessage'
            ));
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }

    public function dailyCheckIn()
    {
        try {

            $user = Auth::user();
            $tasks = Task::where(['status' => 1])->get();
			
			$now = now();
			$wStartDate = $now->copy()->startOfWeek();
			$days[] = $wStartDate->format('Y-m-d');
			$days[] = $wStartDate->addDay(1)->format('Y-m-d');
			$days[] = $wStartDate->addDay(1)->format('Y-m-d');
			$days[] = $wStartDate->addDay(1)->format('Y-m-d');
			$days[] = $wStartDate->addDay(1)->format('Y-m-d');
			$days[] = $wStartDate->addDay(1)->format('Y-m-d');
			$days[] = $now->copy()->endOfWeek()->format('Y-m-d');
			$todayDate = now()->format('Y-m-d');
			
			$c = array_intersect($days, [$todayDate]);
			$todayKey = array_keys($c);
			
			$checkIns = DB::table('daily_check_ins')->selectRaw('DATE_FORMAT(created_at, "%Y-%m-%d") as created_at')->where('user_id', $user->id)->where('created_at','>=', $days[0])->pluck('created_at')->toArray();
            $pageTitle = 'Active Users Tasks';
            $emptyMessage = 'Sorry no any tasks to show';

            return view($this->activeTemplate. 'task.all', compact(
                'pageTitle',
                'user',
                'days',
                'tasks',
                'todayKey',
                'checkIns',
                'todayDate',
                'emptyMessage'
            ));
        } catch (\Exception $th) {
            $notify[] = ['error', $th->getMessage()];
            return redirect()->intended(route('user.home'))->withNotify($notify);
        }
    }

    function usdtDeposit(){
        $pageTitle = "USDT BEP-20 Funding";
        $user = Auth::user();
        $amount = session()->get('fund_amount');
        if(!$amount){
            $amount = 1000;
        }
        $usdt_amount = $amount;

        /*if (!$user->wallet) {
            $notify[] = ['error', 'Service not available, try again latter'];
            return back()->withNotify($notify);
        }*/

        return view($this->activeTemplate. 'user.manual_payment.usdt_process', compact('pageTitle','user','amount','usdt_amount'));

        //return view($this->activeTemplate. 'user.payment.usdt', compact('pageTitle','user'));
    }


    function usdtDepositProccess(Request $request){
        $pageTitle = "USDT Funding";
        $user = Auth::user();
        $amount = $request->amount;
        $usdt_amount = $amount;

        if (!$user->wallet) {
            $notify[] = ['error', 'Service not available, try again latter'];
            return $this->handleResponse($request, 'Service not available, try again latter', 401, $notify);
        }
        
        return view($this->activeTemplate. 'user.manual_payment.usdt_process', compact('pageTitle','user','amount','usdt_amount'));
    }

    public function profile()
    {
        $pageTitle = "Profile Setting";
        $user = Auth::user();
        return view($this->activeTemplate. 'user.profile_setting', compact('pageTitle','user'));
    }

    public function submitProfile(Request $request)
    {
        $request->validate([
            'fullname' => 'sometimes|required|string|max:50',
            'email' => 'sometimes|required|string|max:50',
            'address' => 'sometimes|required|max:80',
            'state' => 'sometimes|required|max:80',
            'zip' => 'sometimes|required|max:40',
            'city' => 'sometimes|required|max:50',
            'image' => ['image',new FileTypeValidate(['jpg','jpeg','png'])]
        ]); $emailError = '';

        $user = Auth::user();
		if( $request->firstname ){
			$in['firstname'] = $request->firstname;
		}
		
		if( $request->lastname ){
			$in['lastname'] = $request->lastname;
		}
		
		if( $request->fullname ){
			$AName = str_replace('  ', ' ', $request->fullname);
			$pAName = explode(' ', $AName );

			if ( isset($pAName[1]) && $pAName[1] ) {
				$in['firstname'] = trim($pAName[0]);
				if( isset($pAName[2]) && $pAName[2] ){
					$in['lastname']  = (strlen($pAName[2]) > strlen($pAName[1])) ? trim($pAName[2]):trim($pAName[1]);
				}else{
					$in['lastname']  = trim($pAName[1]);
				}
			}else{
				$in['firstname'] = trim($pAName[0]);
				$in['lastname']  = substr($request->fullname, 0, 1);
			}
		}
		
		if( $request->email ){

            // Find Email
            $email = User::where('email', $request->email)->first();

            // Exist email
            if($email && $user->email != $email->email) {
                $emailError = 'But email already taken, try new one';
            }else{
				$user->email = $request->email;
			}
		}
		
		if( $request->state ){
			$in['address'] = [
				'address' => $request->address,
				'state' => $request->state,
				'zip' => $request->zip,
				'country' => $request->country,
				'city' => $request->city,
			];
		}elseif( $request->address ){
			$address = $user->address;
			$in['address'] = [
				'address' => $request->address,
				'state' => $address->state,
				'zip' => $address->zip,
				'country' => $address->country,
				'city' => $address->city,
			];
		}

        if ($request->hasFile('image')) {
            $location = imagePath()['profile']['user']['path'];
            $size = imagePath()['profile']['user']['size'];
            $filename = uploadImage($request->image, $location, $size, $user->image);
            $in['image'] = $filename;
        }
        $user->fill($in)->save();
        $notify[] = ['success', 'Profile updated successfully. ' . $emailError];
        return back()->withNotify($notify);
    }

    function profileUpdateEmail(Request $request){
        
        if($request->method() == 'POST') {
            // Validate
            $validator = Validator::make($request->all(),[
                'email' => 'required|email'
            ]);

            // Failed to validate
            if ($validator->fails()) {
                $notify[] = ['error', implode('..', $validator->errors()->all())];
                return back()->withNotify($notify);
            }

            // User
            $user = User::where('email', auth()->user()->email)->first();

            // Find Email
            $email = User::where('email', $request->email)->first();

            // Exist email
            if($email && $user->email != $email->email) {
                // Error
                $notify[] = ['error', 'Email address already taken with other, try new one'];
                return back()->withNotify($notify);
            }

            // Update
            $user->email = $request->email;
            $user->save();


            $notify[] = ['success', 'Email update successfully'];
            return back()->withNotify($notify);
        }
        
        // Return Response
        $pageTitle = "Change Email";
        return view($this->activeTemplate. 'user.profile_update_email', compact('pageTitle'));

    }

    function updateWithdrawalPassword(Request $request){
        
        if($request->method() == 'POST') {
            
            $password_validation = Password::min(6);

            // Validate
            $validator = Validator::make($request->all(),[
                'current_password' => 'required',
                'password' => ['required','confirmed',$password_validation]
            ]);

            // Failed to validate
            if ($validator->fails()) {
                $errors = $validator->errors()->all();
                $notify[] = ['error', implode(' ', $errors)];
                return $this->handleResponse($request, implode(' ', $errors), 401, $notify);
            }
            
            try {
                
                // User
                $user = User::where('email', auth()->user()->email)->first();

                // Verify Password
                if (Hash::check($request->current_password, $user->password)) {
                    $password = Hash::make($request->password);
                    $user->withdrawal_password = $password;
                    $user->save();

                    $notify[] = ['success', 'Withdrawal Password changed successfully.'];
                    return $this->handleResponse($request, 'Password changed successfully', 200, $notify);
                } else {
                    $notify[] = ['error', "The login password doesn't match!"];
                    return $this->handleResponse($request, "The login password doesn't match!", 401, $notify);
                }
            } catch (\PDOException $e) {
                $notify[] = ['error', $e->getMessage()];
                return $this->handleResponse($request, $e->getMessage(), 401, $notify);
            }
        }
        
        // Return Response
        $pageTitle = "Withdrawal Password";
        return view($this->activeTemplate. 'user.withdraw.change_password', compact('pageTitle'));

    }

    public function changePassword()
    {
        $pageTitle = 'Change Password';
        return view($this->activeTemplate . 'user.password', compact('pageTitle'));
    }

    public function submitPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'password' => ['required', 'confirmed', 'min:6']
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $notify[] = ['error', implode(' ', $errors)];
            return $this->handleResponse($request, implode(' ', $errors), 401, $notify);
        }

        try {
            $user = auth()->user();
            if (Hash::check($request->current_password, $user->password)) {
                $password = Hash::make($request->password);
                $user->password = $password;
                $user->save();

                $notify[] = ['success', 'Password changed successfully.'];
                return $this->handleResponse($request, 'Password changed successfully', 200, $notify);
            } else {
                $notify[] = ['error', 'The password doesn\'t match!'];
                return $this->handleResponse($request, 'The password doesn\'t match!', 401, $notify);
            }
        } catch (\PDOException $e) {
            $notify[] = ['error', $e->getMessage()];
            return $this->handleResponse($request, $e->getMessage(), 401, $notify);
        }
    }

    /*
     * Withdraw Operation
     */

    public function withdrawMoney()
    {

        $user = auth()->user();

        $currency = currency( config('settings')->cur_text );

        if(!$currency) return back()->withNotify([['error', 'Currency not support']]);

        $userBalance = $this->wallet->getBalanceRaw($user, $currency->cur_text);

        if($userBalance instanceOf \Exception) {
            $notify[] = ['error', $userBalance->getMessage()];
            return back()->withNotify($notify);
        }

        // If NGN empty
        if(empty($user->bankAccount) && config('settings')->cur_text == 'NGN') {
            $notify[] = ['error', 'Please add your bank account'];
            return redirect()->intended(route('user.account-setup'))->withNotify($notify);
        }

        // If USDT empty
        if(empty($user->withdrawal_usdt_address) && config('settings')->cur_text == 'USDT') {
            $notify[] = ['error', 'Please add your USDT BEP-20 Address'];
            return redirect()->intended(route('user.usdt-setup'))->withNotify($notify);
        }

        $invests = Investment::where('user_id', $user->id)->count();

        if($invests < 1){
            $notify[] = ['error', 'Kindly invest before placing withdrawal'];
            return redirect()->intended(route('user.home'))->withNotify($notify);
        }

        $withdrawMethod = WithdrawMethod::where(['status' => 1, 'currency' => config('settings')->cur_text])
                ->orderBy('name')
                ->get();

        if(empty($withdrawMethod)) {
            $notify[] = ['error', 'Withdrawal not available. Try again later'];
            return redirect()->intended(route('user.home'))->withNotify($notify);
        }

        //last withdrawal
		if( config('settings')->once_withdrawal_aday ){
			$can_withdraw = 1;
			$last = Withdrawal::where('user_id', $user->id)
				->latest()
				->first();

			if(!$last){
				$last_time = Carbon::parse($user->created_at);
				$now = Carbon::now();

				$difference = $last_time->diffInHours($now);

				if($difference >= 24){
					$can_withdraw = 1;
				}else{
					$can_withdraw = 0;
				}
			}else{
				//check last withdrawal
				$last_time = Carbon::parse($last->created_at);
				$now = Carbon::now();

				$difference = $last_time->diffInHours($now);

				if($difference >= 24){
					$can_withdraw = 1;
				}else{
					$can_withdraw = 0;
				}
			}

			if( !$can_withdraw ){
				$notify[] = ['error', "Withdrawals is restricted to once every 24hours, your next withdrawal is in ". (24 - $difference) ." hours time"];
				return back()->withNotify($notify);
			}

		}

		$is_restricted = 1;
		$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";
		if( config('settings')->withdraw_time ){

			$currentTime = Carbon::now('Africa/Lagos');

			$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
			$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

			// It is between 7am and 7pm
			if ($currentTime->between($startTime, $endTime)) {
				$is_restricted = 0;
			}

			if($is_restricted){
				$notify[] = ['error', $restricted_message];
				return redirect()->intended(route('user.home'))->withNotify($notify);
			}
		}


        //return $withdrawMethod;
        $pageTitle = 'Withdraw Money';
        return view($this->activeTemplate.'user.withdraw.request', compact('pageTitle','currency','userBalance','withdrawMethod', 'is_restricted', 'restricted_message','user'));
    }

    public function withdrawStore(Request $request, PaymentServices $payment)
    {
        // Validate
        $validator = Validator::make($request->all(),[
            'amount' => 'required|numeric',
            'method' => 'required|string',
            'debit_from' => 'required|string'
        ]);

        // Failed to validate
        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $notify[] = ['error', implode(' ', $errors)];
            return $this->handleResponse($request, implode(' ', $errors), 401, $notify);
        }

        $currency = ($request->currency) ? $request->currency : config('settings')->cur_text;
		$user = auth()->user();

        try {

			if(!$currency) throw new \Exception("currency not supported");

			if(config('settings')->withdrawal_active == 0) throw new \Exception("Withdrawal not available at the moment");
			
            $method = WithdrawMethod::where('id', $request->method)->where('status', 1)->first();
            
            if (!$method ) throw new \Exception("Withdrawal method not available");

			if( $user->withdrawal_password ){
				// Verify Password
				if (!Hash::check($request->password, $user->withdrawal_password)) throw new \Exception("Withdrawal password doesn't match!");
			}
            
            // Minimum Amount
            if ($request->amount < $method->min_limit) throw new \Exception("Your requested amount is smaller than minimum amount of ".$currency." ".$method->min_limit."");
            
            // Maximum Amount
            if ($request->amount > $method->max_limit) throw new \Exception("Your requested amount is larger than maximum amount of ".$currency." ".$method->max_limit."");
            

            $pend = Withdrawal::where('user_id', $user->id)->where('status', '2')->first();
            if ($pend) {
                $notify[] = ['error', 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again'];
                return $this->handleResponse($request, 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again', 401, $notify);
            }

            // Get Balance
            $userBalance = $this->wallet->getAvailableBalance($user, $request->debit_from, $currency);

            // Exception
            if($userBalance instanceOf \Exception) throw new \Exception($userBalance->getMessage());


            // get users investment
			$invests = Investment::where('user_id', $user->id)->count();
            // Exception
			if($invests < 1) throw new \Exception('Kindly invest before placing withdrawal');
			
			if( config('settings')->withdraw_time ){
				$currentTime = Carbon::now('Africa/Lagos');
				$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
				$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

				$is_restricted = 1;
				$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";

				// It is between 7am and 7pm
				if ($currentTime->between($startTime, $endTime)) {
					$is_restricted = 0;
				}
				if( $is_restricted ) throw new \Exception($restricted_message);
			}

			//last withdrawal
			if( config('settings')->once_withdrawal_aday ){
				$can_withdraw = 1;
				$last = Withdrawal::where('user_id', $user->id)
					->latest()
					->first();

				if(!$last){
					$last_time = Carbon::parse($user->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}else{
					//check last withdrawal
					$last_time = Carbon::parse($last->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}

				if( !$is_restricted ) throw new \Exception("Withdrawals is restricted to once every 24hours, your next withdrawal is in ". (24 - $difference) ." hours time");
			}
			
			
			// All verification passed
            $charge = $method->fixed_charge + ($request->amount * $method->percent_charge / 100);

            $afterCharge = $request->amount - $charge;
            $finalAmount = round($afterCharge / $method->rate);

            // Debit Balance
            $debitBalance = $this->wallet->debitAvailableBalance($user, $request->debit_from, $currency, $request->amount);

            if($debitBalance instanceOf \Exception) throw new \Exception($debitBalance->getMessage());

            $reference = getTrx();
            $reqField = [];
			$hasAuto = true;

            // Add bank info
            if($method->is_bank) {
				// if( in_array($user->bankAccount->bank_code, ['090267','100033','100026','100004','090405','NR0186','NR0223','NR0096']) ){
				// 	$hasAuto = false;
				// }
                $reqField["bank_name"] = [
                    'field_name' => $user->bankAccount->bank_name,
                    'type' => "text"
                ];

                $reqField["account_name"] = [
                    'field_name' => $user->bankAccount->account_name,
                    'type' => "text"
                ];

                $reqField["account_number"] = [
                    'field_name' => $user->bankAccount->account_number,
                    'type' => "text"
                ];
            }else {
                // USDT 
                $reqField["address"] = [
                    'field_name' => $user->withdrawal_usdt_address,
                    'type' => "text"
                ];
            }
            
            $statusData = 2;
            // Add Withdrawal
            $withdraw = new Withdrawal();

            // Auto Deposit
            if(config('settings')->auto_transfer && $hasAuto ) {
				
                // Transfer Payment
                $transferPayment = $payment->payout($reference, 'NGN', $finalAmount, $method->gateway->alias, $user->bankAccount->bank_code, $user->bankAccount->account_number, $user->bankAccount->account_name);
				
                // Exception
                if($transferPayment['status'] == false) {

                    // Refund Payment
                    $this->wallet->creditAvailableBalance($user, $request->debit_from, $currency, $request->amount);

                    // Update Transaction
                    self::addUpdateTransaction('withdraw', showAmount($finalAmount) . ' ' . $method->currency . " Withdraw Via " . $method->name, $reference, 3);
                    
                    $notify[] = ['error', $transferPayment['message']];
                    return $this->handleResponse($request, $transferPayment['message'], 401, $notify);
                }
				
                if( @$transferPayment['order_ref'] ) {
					$withdraw->wallet = $user['order_ref'];
				}

                $statusData = 1;
            }
            
            $account_info = [
                'bank_account' => $user->bankAccount->account_number,
                'full_name' => $user->bankAccount->account_name,
                'bank_name' => $user->bankAccount->bank_name,
                'bank_code' => $user->bankAccount->bank_code,
            ];
            
            $withdraw->method_id = $method->id; // wallet method ID
            $withdraw->user_id = $user->id;
            $withdraw->amount = $request->amount;
            $withdraw->currency = $method->currency;
            $withdraw->rate = $method->rate;
            $withdraw->charge = $charge;
            $withdraw->final_amount = $finalAmount;
            $withdraw->after_charge = $afterCharge;
            $withdraw->is_bank = $method->is_bank;
            $withdraw->trx = $reference;
            $withdraw->charge_type = $request->debit_from;
            $withdraw->status = $statusData;
            $withdraw->account_info = json_encode($account_info);
            $withdraw['withdraw_information'] = $reqField;
            $withdraw->save();
            
            // Update Transaction
            self::addUpdateTransaction('withdraw', showAmount($withdraw->final_amount) . ' ' . $withdraw->currency . " Withdraw Via " . $withdraw->method->name, $withdraw->trx, $statusData);

            $adminNotification = new AdminNotification();
            $adminNotification->user_id = $user->id;
            $adminNotification->title = "New withdraw request from ".$user->username;
            $adminNotification->click_url = urlPath('admin.withdraw.details',$withdraw->id);
            $adminNotification->save();

            session()->put("withdrawal_code","-");

            //create bank withdrawal

            if($withdraw->is_bank) {
                $bank_withdrawal = new BankWithdrawal();
                $bank_withdrawal->withdrawal_id = $withdraw->id;
                $bank_withdrawal->bank_code = $user->bankAccount->bank_code;
                $bank_withdrawal->account_number = $user->bankAccount->account_number;
                $bank_withdrawal->account_name = $user->bankAccount->account_name;
                $bank_withdrawal->save();
            }


            $message = Split_Hide_Name($user->username)." withdraw ".amount_format($withdraw->final_amount);
            $board = new Board();
            $board->messages = $message;
            $board->save();

            notify($user, 'WITHDRAW_REQUEST', [
                'method_name' => $withdraw->method->name,
                'method_currency' => $withdraw->currency,
                'method_amount' => showAmount($withdraw->final_amount),
                'amount' => showAmount($withdraw->amount),
                'charge' => showAmount($withdraw->charge),
                'currency' => $withdraw->currency,
                'rate' => showAmount($withdraw->rate),
                'trx' => $withdraw->trx,
                'post_balance' => showAmount($userBalance),
                'delay' => $withdraw->method->delay
            ]);

            // Return Response
            return $this->handleResponseRedirect($request, 'Withdraw request sent successfully', 200, urlPath('user.transaction.type', ['type' => 'withdraw']));
            
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return $this->handleResponse($request, $th->getMessage(), 401, $notify);
        }
    }

    /*
     * Withdraw Operation
     */

    public function withdrawReferral()
    {

        $user = auth()->user();

        $currency = currency( $user->user_currency );

        if(!$currency) return back()->withNotify([['error', 'Currency not support']]);

        // If USDT empty
        if(empty($user->withdrawal_usdt_address) && $currency['cur_text'] == 'USDT') {
            $notify[] = ['error', 'Please add your USDT BEP-20 Address'];
            return redirect()->intended(route('user.usdt-setup'))->withNotify($notify);
        }

        // If NGN empty
        if(empty($user->bankAccount)) {
			if( $currency['cur_text'] == 'NGN') {
				$notify[] = ['error', 'Please add your bank account'];
				return redirect()->intended(route('user.account-setup'))->withNotify($notify);
			}else{
				$notify[] = ['error', 'Please add your payment account details'];
				return redirect()->intended(route('user.mobile-money'))->withNotify($notify);
			}
		}

        if(empty($user->referral_wallet )){
            $notify[] = ['error', 'You do not have any pending referral balance to withdraw from.'];
            return redirect()->intended(route('user.profile'))->withNotify($notify);
        }

        $withdrawMethod = WithdrawMethod::where(['status' => 1, 'currency' => $currency['cur_text']])
                ->orderBy('name')
                ->get();

        if(empty($withdrawMethod)) {
            $notify[] = ['error', 'Withdrawal not available for '. $currency['cur_text'] .' currency at the moment.'];
            return redirect()->intended(route('user.profile'))->withNotify($notify);
        }

		$is_restricted = 1;
		$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";
		if( config('settings')->withdraw_time ){

			$currentTime = Carbon::now('Africa/Lagos');

			$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
			$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

			// It is between 7am and 7pm
			if ($currentTime->between($startTime, $endTime)) {
				$is_restricted = 0;
			}

			if($is_restricted){
				$notify[] = ['error', $restricted_message];
				return redirect()->intended(route('user.profile'))->withNotify($notify);
			}
		}

		$totalWithdraw = Withdrawal::where('user_id', $user->id)->where('wallet', 'referral')->sum('amount');

        //return $withdrawMethod;
        $pageTitle = 'Referral withdrawal';
        return view($this->activeTemplate.'user.withdraw.referral', compact('pageTitle','currency','withdrawMethod', 'user', 'totalWithdraw'));
    }

    public function withdrawRefStore(Request $request, PaymentServices $payment)
    {
        // Validate
        $validator = Validator::make($request->all(),[
            'amount' => 'required|numeric',
            'method' => 'required|string',
            'debit_from' => 'required|string'
        ]);

        // Failed to validate
        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $notify[] = ['error', implode(' ', $errors)];
            return $this->handleResponse($request, implode(' ', $errors), 401, $notify);
        }

        $currency = ($request->currency) ? $request->currency : config('settings')->cur_text;
		$user = auth()->user();

        try {

			if(!$currency) throw new \Exception("currency not supported");

			if(config('settings')->withdrawal_active == 0) throw new \Exception("Withdrawal not available at the moment");
			
            $method = WithdrawMethod::where('id', $request->method)->where('status', 1)->first();
            
            if (!$method ) throw new \Exception("Withdrawal method not available");

			if( $user->withdrawal_password ){
				// Verify Password
				if (!Hash::check($request->password, $user->withdrawal_password)) throw new \Exception("Withdrawal password doesn't match!");
			}
            
            // Minimum Amount
            if ($request->amount < $method->min_limit) throw new \Exception("Your requested amount is smaller than minimum amount of ".$currency." ".$method->min_limit."");
            
            // Maximum Amount
            if ($request->amount > $method->max_limit) throw new \Exception("Your requested amount is larger than maximum amount of ".$currency." ".$method->max_limit."");
            

            $pend = Withdrawal::where('user_id', $user->id)->where('status', '2')->first();
            if ($pend) {
                $notify[] = ['error', 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again'];
                return $this->handleResponse($request, 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again', 401, $notify);
            }

            // Get Balance
            $userBalance = $this->wallet->getUserBalance($user, $request->debit_from, $currency);

            // Exception
            if($userBalance instanceOf \Exception) throw new \Exception($userBalance->getMessage());


            // get users investment
			$invests = Investment::where('user_id', $user->id)->count();
            // Exception
			if($invests < 1) throw new \Exception('Kindly invest before placing withdrawal');
			
			if( config('settings')->withdraw_time ){
				$currentTime = Carbon::now('Africa/Lagos');
				$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
				$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

				$is_restricted = 1;
				$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";

				// It is between 7am and 7pm
				if ($currentTime->between($startTime, $endTime)) {
					$is_restricted = 0;
				}
				if( $is_restricted ) throw new \Exception($restricted_message);
			}

			//last withdrawal
			if( config('settings')->once_withdrawal_aday ){
				$can_withdraw = 1;
				$last = Withdrawal::where('user_id', $user->id)
					->latest()
					->first();

				if(!$last){
					$last_time = Carbon::parse($user->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}else{
					//check last withdrawal
					$last_time = Carbon::parse($last->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}

				if( !$is_restricted ) throw new \Exception("Withdrawals is restricted to once every 24hours, your next withdrawal is in ". (24 - $difference) ." hours time");
			}
			
			
			// All verification passed
            $charge = $method->fixed_charge + ($request->amount * $method->percent_charge / 100);

            $afterCharge = $request->amount - $charge;
            $finalAmount = round($afterCharge / $method->rate);

            // Debit Balance
            $debitBalance = $this->wallet->debitAvailableBalance($user, $request->debit_from, $currency, $request->amount);

            if($debitBalance instanceOf \Exception) throw new \Exception($debitBalance->getMessage());

            $reference = getTrx();
            $reqField = [];
			$hasAuto = true;

            // Add bank info
            if($method->is_bank) {
				// if( in_array($user->bankAccount->bank_code, ['090267','100033','100026','100004','090405','NR0186','NR0223','NR0096']) ){
				// 	$hasAuto = false;
				// }
                $reqField["bank_name"] = [
                    'field_name' => $user->bankAccount->bank_name,
                    'type' => "text"
                ];

                $reqField["account_name"] = [
                    'field_name' => $user->bankAccount->account_name,
                    'type' => "text"
                ];

                $reqField["account_number"] = [
                    'field_name' => $user->bankAccount->account_number,
                    'type' => "text"
                ];
            }else {
                // USDT 
                $reqField["address"] = [
                    'field_name' => $user->withdrawal_usdt_address,
                    'type' => "text"
                ];
            }
            
            $statusData = 2;

            // Add Withdrawal
            $withdraw = new Withdrawal();

            // Auto Deposit
            if(config('settings')->auto_transfer && $hasAuto ) {
                // Transfer Payment
                $transferPayment = $payment->payout($reference, 'NGN', $finalAmount, $method->gateway->alias, $user->bankAccount->bank_code, $user->bankAccount->account_number, $user->bankAccount->account_name);

                // Exception
                if($transferPayment['status'] == false) {

                    // Refund Payment
                    $this->wallet->creditAvailableBalance($user, $request->debit_from, $currency, $request->amount);

                    // Update Transaction
                    self::addUpdateTransaction('withdraw', showAmount($finalAmount) . ' ' . $method->currency . " Withdraw Via " . $method->name, $reference, 3);
                    
                    $notify[] = ['error', $transferPayment['message']];
                    return $this->handleResponse($request, $transferPayment['message'], 401, $notify);
                }

                if( @$transferPayment['order_ref'] ) {
					$withdraw->wallet = $user['order_ref'];
				}

                $statusData = 1;
            }
            
            $account_info = [
                'bank_account' => $user->bankAccount->account_number,
                'full_name' => $user->bankAccount->account_name,
                'bank_name' => $user->bankAccount->bank_name,
                'bank_code' => $user->bankAccount->bank_code,
            ];
            
            $withdraw->method_id = $method->id; // wallet method ID
            $withdraw->user_id = $user->id;
            $withdraw->amount = $request->amount;
            $withdraw->currency = $method->currency;
            $withdraw->rate = $method->rate;
            $withdraw->charge = $charge;
            $withdraw->final_amount = $finalAmount;
            $withdraw->after_charge = $afterCharge;
            $withdraw->is_bank = $method->is_bank;
            $withdraw->trx = $reference;
            $withdraw->wallet = 'referral';
            $withdraw->charge_type = $request->debit_from;
            $withdraw->status = $statusData;
            $withdraw->account_info = json_encode($account_info);
            $withdraw['withdraw_information'] = $reqField;
            $withdraw->save();
            
            // Update Transaction
            self::addUpdateTransaction('withdraw', showAmount($withdraw->final_amount) . ' ' . $withdraw->currency . " Withdraw Via " . $withdraw->method->name, $withdraw->trx, $statusData);

            $adminNotification = new AdminNotification();
            $adminNotification->user_id = $user->id;
            $adminNotification->title = "New withdraw request from ".$user->username;
            $adminNotification->click_url = urlPath('admin.withdraw.details',$withdraw->id);
            $adminNotification->save();

            session()->put("withdrawal_code","-");

            //create bank withdrawal

            if($withdraw->is_bank) {
                $bank_withdrawal = new BankWithdrawal();
                $bank_withdrawal->withdrawal_id = $withdraw->id;
                $bank_withdrawal->bank_code = $user->bankAccount->bank_code;
                $bank_withdrawal->account_number = $user->bankAccount->account_number;
                $bank_withdrawal->account_name = $user->bankAccount->account_name;
                $bank_withdrawal->save();
            }


            $message = Split_Hide_Name($user->username)." withdraw ".amount_format($withdraw->final_amount);
            $board = new Board();
            $board->messages = $message;
            $board->save();

            notify($user, 'WITHDRAW_REQUEST', [
                'method_name' => $withdraw->method->name,
                'method_currency' => $withdraw->currency,
                'method_amount' => showAmount($withdraw->final_amount),
                'amount' => showAmount($withdraw->amount),
                'charge' => showAmount($withdraw->charge),
                'currency' => $withdraw->currency,
                'rate' => showAmount($withdraw->rate),
                'trx' => $withdraw->trx,
                'post_balance' => showAmount($userBalance),
                'delay' => $withdraw->method->delay
            ]);

            // Return Response
            return $this->handleResponseRedirect($request, 'Withdraw request sent successfully', 200, urlPath('user.transaction.type', ['type' => 'withdraw']));
            
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return $this->handleResponse($request, $th->getMessage(), 401, $notify);
        }
    }


    /*
     * Withdraw Operation
     */

    public function withdrawBonus()
    {

        $user = auth()->user();

        $currency = currency( $user->user_currency );

        if(!$currency) return back()->withNotify([['error', 'Currency not support']]);

        // If USDT empty
        if(empty($user->withdrawal_usdt_address) && $currency['cur_text'] == 'USDT') {
            $notify[] = ['error', 'Please add your USDT BEP-20 Address'];
            return redirect()->intended(route('user.usdt-setup'))->withNotify($notify);
        }

        // If NGN empty
        if(empty($user->bankAccount)) {
			if( $currency['cur_text'] == 'NGN') {
				$notify[] = ['error', 'Please add your bank account'];
				return redirect()->intended(route('user.account-setup'))->withNotify($notify);
			}else{
				$notify[] = ['error', 'Please add your payment account details'];
				return redirect()->intended(route('user.mobile-money'))->withNotify($notify);
			}
		}

        if(empty($user->bonus_balance )){
            $notify[] = ['error', 'You do not have any bonus balance to withdraw.'];
            return redirect()->intended(route('user.profile'))->withNotify($notify);
        }

        $withdrawMethod = WithdrawMethod::where(['status' => 1, 'currency' => $currency['cur_text']])
                ->orderBy('name')
                ->get();

        if(empty($withdrawMethod)) {
            $notify[] = ['error', 'Withdrawal not available for '. $currency['cur_text'] .' currency at the moment.'];
            return redirect()->intended(route('user.profile'))->withNotify($notify);
        }

		$is_restricted = 1;
		$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";
		if( config('settings')->withdraw_time ){

			$currentTime = Carbon::now('Africa/Lagos');

			$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
			$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

			// It is between 7am and 7pm
			if ($currentTime->between($startTime, $endTime)) {
				$is_restricted = 0;
			}

			if($is_restricted){
				$notify[] = ['error', $restricted_message];
				return redirect()->intended(route('user.profile'))->withNotify($notify);
			}
		}

		$totalWithdraw = Withdrawal::where('user_id', $user->id)->where('wallet', 'bonus')->sum('amount');

        //return $withdrawMethod;
        $pageTitle = 'Bonus withdrawal';
        return view($this->activeTemplate.'user.withdraw.bonus', compact('pageTitle','currency','withdrawMethod', 'user', 'totalWithdraw'));
    }

    public function withdrawBonusStore(Request $request, PaymentServices $payment)
    {
        // Validate
        $validator = Validator::make($request->all(),[
            'amount' => 'required|numeric',
            'method' => 'required|string',
            'debit_from' => 'required|string'
        ]);

        // Failed to validate
        if ($validator->fails()) {
            $errors = $validator->errors()->all();
            $notify[] = ['error', implode(' ', $errors)];
            return $this->handleResponse($request, implode(' ', $errors), 401, $notify);
        }

		$user = auth()->user();
        $currency = ($request->currency) ? $request->currency : $user->user_currency;

        try {

			if(!$currency) throw new \Exception("currency not supported");

			if(config('settings')->withdrawal_active == 0) throw new \Exception("Withdrawal not available at the moment");
			
            $method = WithdrawMethod::where('id', $request->method)->where('status', 1)->first();
            
            if (!$method ) throw new \Exception("Withdrawal method not available");

			if( $user->withdrawal_password ){
				// Verify Password
				if (!Hash::check($request->password, $user->withdrawal_password)) throw new \Exception("Withdrawal password doesn't match!");
			}
            
            // Minimum Amount
            if ($request->amount < $method->min_limit) throw new \Exception("Your requested amount is smaller than minimum amount of ".$currency." ".$method->min_limit."");
            
            // Maximum Amount
            if ($request->amount > $method->max_limit) throw new \Exception("Your requested amount is larger than maximum amount of ".$currency." ".$method->max_limit."");
            

            $pend = Withdrawal::where('user_id', $user->id)->where('status', '2')->first();
            if ($pend) {
                $notify[] = ['error', 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again'];
                return $this->handleResponse($request, 'You currently have a pending withdrawal, please wait for confirmation or cancellation and try again', 401, $notify);
            }

            // Get Balance
            $userBalance = $this->wallet->getUserBalance($user, $request->debit_from, $currency);

            // Exception
            if($userBalance instanceOf \Exception) throw new \Exception($userBalance->getMessage());


            // get users investment
			$invests = Investment::where('user_id', $user->id)->count();
            // Exception
			if($invests < 1) throw new \Exception('Kindly invest before placing withdrawal');
			
			if( config('settings')->withdraw_time ){
				$currentTime = Carbon::now('Africa/Lagos');
				$startTime = Carbon::parse('8:00:00', 'Africa/Lagos');
				$endTime = Carbon::parse('22:00:00', 'Africa/Lagos');

				$is_restricted = 1;
				$restricted_message = "Sorry! Withdrawal time are between 8:00 AM to 10:00 PM Nigerian Time";

				// It is between 7am and 7pm
				if ($currentTime->between($startTime, $endTime)) {
					$is_restricted = 0;
				}
				if( $is_restricted ) throw new \Exception($restricted_message);
			}

			//last withdrawal
			if( config('settings')->once_withdrawal_aday ){
				$can_withdraw = 1;
				$last = Withdrawal::where('user_id', $user->id)
					->latest()
					->first();

				if(!$last){
					$last_time = Carbon::parse($user->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}else{
					//check last withdrawal
					$last_time = Carbon::parse($last->created_at);
					$now = Carbon::now();

					$difference = $last_time->diffInHours($now);

					if($difference >= 24){
						$can_withdraw = 1;
					}else{
						$can_withdraw = 0;
					}
				}

				if( !$is_restricted ) throw new \Exception("Withdrawals is restricted to once every 24hours, your next withdrawal is in ". (24 - $difference) ." hours time");
			}
			
			
			// All verification passed
            $charge = $method->fixed_charge + ($request->amount * $method->percent_charge / 100);

            $afterCharge = $request->amount - $charge;
            $finalAmount = round($afterCharge / $method->rate);

            // Debit Balance
            $debitBalance = $this->wallet->debitAvailableBalance($user, $request->debit_from, $currency, $request->amount);

            if($debitBalance instanceOf \Exception) throw new \Exception($debitBalance->getMessage());

            $reference = getTrx();
            $reqField = [];
			$hasAuto = true;

            // Add bank info
            if($method->is_bank) {
				// if( in_array($user->bankAccount->bank_code, ['090267','100033','100026','100004','090405','NR0186','NR0223','NR0096']) ){
				// 	$hasAuto = false;
				// }
				
                $reqField["bank_name"] = [
                    'field_name' => $user->bankAccount->bank_name,
                    'type' => "text"
                ];

                $reqField["account_name"] = [
                    'field_name' => $user->bankAccount->account_name,
                    'type' => "text"
                ];

                $reqField["account_number"] = [
                    'field_name' => $user->bankAccount->account_number,
                    'type' => "text"
                ];
            }else {
                // USDT 
                $reqField["address"] = [
                    'field_name' => $user->withdrawal_usdt_address,
                    'type' => "text"
                ];
            }
            
            $statusData = 2;
			
            // Add Withdrawal
            $withdraw = new Withdrawal();

            // Auto Deposit
            if(config('settings')->auto_transfer && $hasAuto ) {
                // Transfer Payment
                $transferPayment = $payment->payout($reference, 'NGN', $finalAmount, $method->gateway->alias, $user->bankAccount->bank_code, $user->bankAccount->account_number, $user->bankAccount->account_name);

                // Exception
                if($transferPayment['status'] == false) {

                    // Refund Payment
                    $this->wallet->creditAvailableBalance($user, $request->debit_from, $currency, $request->amount);

                    // Update Transaction
                    self::addUpdateTransaction('withdraw', showAmount($finalAmount) . ' ' . $method->currency . " Withdraw Via " . $method->name, $reference, 3);
                    
                    $notify[] = ['error', $transferPayment['message']];
                    return $this->handleResponse($request, $transferPayment['message'], 401, $notify);
                }

                if( @$transferPayment['order_ref'] ) {
					$withdraw->wallet = $user['order_ref'];
				}

                $statusData = 1;
            }
            
            $account_info = [
                'bank_account' => $user->bankAccount->account_number,
                'full_name' => $user->bankAccount->account_name,
                'bank_name' => $user->bankAccount->bank_name,
                'bank_code' => $user->bankAccount->bank_code,
            ];
            
            $withdraw->method_id = $method->id; // wallet method ID
            $withdraw->user_id = $user->id;
            $withdraw->amount = $request->amount;
            $withdraw->currency = $method->currency;
            $withdraw->rate = $method->rate;
            $withdraw->charge = $charge;
            $withdraw->final_amount = $finalAmount;
            $withdraw->after_charge = $afterCharge;
            $withdraw->is_bank = $method->is_bank;
            $withdraw->trx = $reference;
            $withdraw->wallet = 'bonus';
            $withdraw->charge_type = $request->debit_from;
            $withdraw->status = $statusData;
            $withdraw->account_info = json_encode($account_info);
            $withdraw['withdraw_information'] = $reqField;
            $withdraw->save();
            
            // Update Transaction
            self::addUpdateTransaction('withdraw', showAmount($withdraw->final_amount) . ' ' . $withdraw->currency . " Withdraw Via " . $withdraw->method->name, $withdraw->trx, $statusData);

            $adminNotification = new AdminNotification();
            $adminNotification->user_id = $user->id;
            $adminNotification->title = "New withdraw request from ".$user->username;
            $adminNotification->click_url = urlPath('admin.withdraw.details',$withdraw->id);
            $adminNotification->save();

            session()->put("withdrawal_code","-");

            //create bank withdrawal

            if($withdraw->is_bank) {
                $bank_withdrawal = new BankWithdrawal();
                $bank_withdrawal->withdrawal_id = $withdraw->id;
                $bank_withdrawal->bank_code = $user->bankAccount->bank_code;
                $bank_withdrawal->account_number = $user->bankAccount->account_number;
                $bank_withdrawal->account_name = $user->bankAccount->account_name;
                $bank_withdrawal->save();
            }


            $message = Split_Hide_Name($user->username)." withdraw ".amount_format($withdraw->final_amount);
            $board = new Board();
            $board->messages = $message;
            $board->save();

            notify($user, 'WITHDRAW_REQUEST', [
                'method_name' => $withdraw->method->name,
                'method_currency' => $withdraw->currency,
                'method_amount' => showAmount($withdraw->final_amount),
                'amount' => showAmount($withdraw->amount),
                'charge' => showAmount($withdraw->charge),
                'currency' => $withdraw->currency,
                'rate' => showAmount($withdraw->rate),
                'trx' => $withdraw->trx,
                'post_balance' => showAmount($userBalance),
                'delay' => $withdraw->method->delay
            ]);

            // Return Response
            return $this->handleResponseRedirect($request, 'Withdraw request sent successfully', 200, urlPath('user.transaction.type', ['type' => 'withdraw']));
            
        } catch (\Exception $th) {
            //throw $th;
            $notify[] = ['error', $th->getMessage()];
            return $this->handleResponse($request, $th->getMessage(), 401, $notify);
        }
    }


    public static function addUpdateTransaction(
        string $type,
        string $desc = null,
        string $reference,
        int $status = 1
    ) {
        try {

            $withdraw = Withdrawal::where('trx', $reference)->first();

            $user = User::find($withdraw->user_id);

            // Find Transaction
            $transaction = Transaction::where('trx',$withdraw->trx)->first();

            // Verify transaction
            if($transaction) {
                //
                $transaction->status = $status;
                $transaction->save();

                return $transaction;
            }

            // Add Transaction
            $transaction = new Transaction();
            $transaction->user_id = $withdraw->user_id;
            $transaction->type =  $type;
            $transaction->amount = $withdraw->amount;
            if($withdraw->is_bonus) {
                $transaction->post_balance = $user->bonus_balance;
            }else{
                $transaction->post_balance = $user->balance;
            }
            $transaction->charge = $withdraw->charge;
            $transaction->currency = $withdraw->currency;
            $transaction->trx_type = '-';
            $transaction->details = $desc;
            $transaction->trx =  $withdraw->trx;
            $transaction->status =  $status;
            $transaction->save();

            return $transaction;
        } catch (\Exception $th) {
            //throw $th;
            return $th;
        }
    }

    public function withdrawPreview()
    {
        $data['withdraw'] = Withdrawal::with('method','user')->where('trx', session()->get('wtrx'))->where('status', 0)->orderBy('id','desc')->firstOrFail();
        $data['pageTitle'] = 'Withdraw Preview';

        if($data['withdraw']->is_bank){
            //$banks = getAllBanks();
            $banks = Bank::all();
            $data['banks'] = $banks;//['data'];
            $data['all_banks'] = json_encode($data['banks']);
            $data['user'] = \auth()->user();

        }



        return view($this->activeTemplate . 'user.withdraw.preview', $data);
    }


    public function withdrawSubmit(Request $request)
    {
        $withdraw = Withdrawal::with('method','user')->where('trx', session()->get('wtrx'))->where('status', 0)->orderBy('id','desc')->firstOrFail();

        $user = User::find(\auth()->user()->id);

        $userBalance = $this->wallet->getBalance($user, $withdraw->currency);

        if($userBalance instanceOf \Exception) {
            $notify[] = ['error', $userBalance->getMessage()];
            return back()->withNotify($notify);
        }

        $rules = [];
        $inputField = [];
        if ($withdraw->method->user_data != null) {
            foreach ($withdraw->method->user_data as $key => $cus) {
                $rules[$key] = [$cus->validation];
                if ($cus->type == 'file') {
                    array_push($rules[$key], 'image');
                    array_push($rules[$key], new FileTypeValidate(['jpg','jpeg','png']));
                    array_push($rules[$key], 'max:2048');
                }
                if ($cus->type == 'text') {
                    array_push($rules[$key], 'max:191');
                }
                if ($cus->type == 'textarea') {
                    array_push($rules[$key], 'max:300');
                }
                $inputField[] = $key;
            }
        }

        $this->validate($request, $rules);

        if($withdraw->is_bonus) {
            if ($withdraw->amount > $user->bonus_balance) {
                $notify[] = ['error', 'Your request amount is larger then your current bonus balance.'];
                return back()->withNotify($notify);
            }
        }else{
            if ($withdraw->amount > $userBalance) {
                $notify[] = ['error', 'Your request amount is larger then your current main balance.'];
                return back()->withNotify($notify);
            }
        }

        /*if($request->otp != session()->get('withdrawal_code')){
            $notify[] = ['error', 'Invalid Verification Code'];
            return back()->withNotify($notify);
        }*/

        //check if i have deposit before

        $before = Investment::where('user_id', $user->id)->first();
        if(!$before){
            $notify[] = ['error', 'You need stock to withdraw'];
            return back()->withNotify($notify);
        }

        $directory = date("Y")."/".date("m")."/".date("d");
        $path = imagePath()['verify']['withdraw']['path'].'/'.$directory;
        $collection = collect($request);
        $reqField = [];
        if ($withdraw->method->user_data != null) {
            foreach ($collection as $k => $v) {
                foreach ($withdraw->method->user_data as $inKey => $inVal) {
                    if ($k != $inKey) {
                        continue;
                    } else {
                        if ($inVal->type == 'file') {
                            if ($request->hasFile($inKey)) {
                                try {
                                    $reqField[$inKey] = [
                                        'field_name' => $directory.'/'.uploadImage($request[$inKey], $path),
                                        'type' => $inVal->type,
                                    ];
                                } catch (\Exception $exp) {
                                    $notify[] = ['error', 'Could not upload your ' . $request[$inKey]];
                                    return back()->withNotify($notify)->withInput();
                                }
                            }
                        } else {
                            $reqField[$inKey] = $v;
                            $reqField[$inKey] = [
                                'field_name' => $v,
                                'type' => $inVal->type,
                            ];
                        }
                    }
                }
            }


            if($withdraw->method->is_bank) {
                $reqField["bank_name"] = [
                    'field_name' => $user->bankAccount->bank_name,
                    'type' => "text"
                ];

                $reqField["account_name"] = [
                    'field_name' => $user->bankAccount->account_name,
                    'type' => "text"
                ];

                $reqField["account_number"] = [
                    'field_name' => $user->bankAccount->account_number,
                    'type' => "text"
                ];
            }

            $withdraw['withdraw_information'] = $reqField;
        } else {
            $withdraw['withdraw_information'] = null;
        }


        $withdraw->status = 2;
        $withdraw->save();
        if($withdraw->is_bonus){
            $user->bonus_balance -= $withdraw->amount;
            $text = " Bonus ";
        }else {
            $userBalance -= $withdraw->amount;
            $text = "";
        }
        $user->save();



        /*$transaction = new Transaction();
        $transaction->user_id = $withdraw->user_id;
        $transaction->type =  'withdraw';
        $transaction->amount = $withdraw->amount;
        $transaction->post_balance = $userBalance;
        $transaction->charge = $withdraw->charge;
        $transaction->trx_type = '-';
        $transaction->details = showAmount($withdraw->final_amount) . ' ' . $withdraw->currency . " $text Withdraw Via " . $withdraw->method->name;
        $transaction->trx =  $withdraw->trx;
        $transaction->status =  2;
        $transaction->save();*/
        self::addUpdateTransaction('withdraw', showAmount($withdraw->final_amount) . ' ' . $withdraw->currency . " Withdraw Via " . $withdraw->method->name, $withdraw->trx, 2);

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title = "New $text withdraw request from ".$user->username;
        $adminNotification->click_url = urlPath('admin.withdraw.details',$withdraw->id);
        $adminNotification->save();


        session()->put("withdrawal_code","-");


        //create bank withdrawal

        if($withdraw->is_bank) {
            $bank_withdrawal = new BankWithdrawal();
            $bank_withdrawal->withdrawal_id = $withdraw->id;
            $bank_withdrawal->bank_code = $user->bankAccount->bank_code;
            $bank_withdrawal->account_number = $user->bankAccount->account_number;
            $bank_withdrawal->account_name = $user->bankAccount->account_name;
            $bank_withdrawal->save();
        }



        $msg = "Dear $user->username, you have successfully requested to sell your dollars worth $withdraw->amount, payment will be disbursed to your account shortly.";



        notify($user, 'WITHDRAW_REQUEST', [
            'method_name' => $withdraw->method->name,
            'method_currency' => $withdraw->currency,
            'method_amount' => showAmount($withdraw->final_amount),
            'amount' => showAmount($withdraw->amount),
            'charge' => showAmount($withdraw->charge),
            'currency' => $withdraw->currency,
            'rate' => showAmount($withdraw->rate),
            'trx' => $withdraw->trx,
            'post_balance' => showAmount($userBalance),
            'delay' => $withdraw->method->delay
        ]);

        $notify[] = ['success', 'Withdraw request sent successfully'];
        return redirect()->route('user.withdraw.history')->withNotify($notify);
    }


    public function show2faForm()
    {
        $ga = new GoogleAuthenticator();
        $user = auth()->user();
        $secret = $ga->createSecret();
		$qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . config('settings')->sitename, $secret);
        $pageTitle = 'Two Factor';
        return view($this->activeTemplate.'user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $user = auth()->user();
        $this->validate($request, [
            'key' => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user,$request->code,$request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts = 1;
            $user->save();
            $userAgent = getIpInfo();
            $osBrowser = osBrowser();
            notify($user, '2FA_ENABLE', [
                'operating_system' => @$osBrowser['os_platform'],
                'browser' => @$osBrowser['browser'],
                'ip' => @$userAgent['ip'],
                'time' => @$userAgent['time']
            ]);
            $notify[] = ['success', 'Google authenticator enabled successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }


    public function disable2fa(Request $request)
    {
        $this->validate($request, [
            'code' => 'required',
        ]);

        $user = auth()->user();
        $response = verifyG2fa($user,$request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts = 0;
            $user->save();
            $userAgent = getIpInfo();
            $osBrowser = osBrowser();
            notify($user, '2FA_DISABLE', [
                'operating_system' => @$osBrowser['os_platform'],
                'browser' => @$osBrowser['browser'],
                'ip' => @$userAgent['ip'],
                'time' => @$userAgent['time']
            ]);
            $notify[] = ['success', 'Two factor authenticator disable successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function rules(){
        $pageTitle = 'Rules';
        return view($this->activeTemplate.'user.rules', compact('pageTitle'));
    }

    function changeCurrencySession($currency = null){
        
        if($currency) {

            // Verify currency
            $verifyCurrency = $this->wallet->verifyCurrency($currency);

            if ($verifyCurrency instanceof \Exception) {
                $notify[] = ['error', $verifyCurrency->getMessage()];
                return back()->withNotify($notify);
            }

            // Set session
            session()->put('currency', $currency);
            session()->put('currency_sym', $verifyCurrency['sym']);

            $notify[] = ['success', 'Currency updated'];
            return back();
        }
        
        $notify[] = ['error', 'Sorry request failed'];
        return back()->withNotify($notify);

    }


    public function referrals(){
        $data['active_ref'] = User::where('ref_by', auth()->user()->id)->whereHas('approveddeposits')->count();
        $data['levels'] = Team::where(['commission_type' => 'invest'])->get();
        $data['user'] = auth()->user();
        $data['referral_count'] = User::where("ref_by", $data['user']->id)->count();

        $data['pageTitle'] = 'Teams';
        return view($this->activeTemplate.'user.referrals', $data);
    }


    public function view_profile($viewID = null){
        $referrals = User::where('ref_by',auth()->user()->id)->paginate(getPaginate());
        $pageTitle = 'My Account';

        $user = User::find(\auth()->user()->id);

        $userBalance = $this->wallet->getBalanceRaw($user, config('settings')->cur_text);

        if($userBalance instanceOf \Exception) {
            $notify[] = ['error', $userBalance->getMessage()];
            return back()->withNotify($notify);
        }

        $totalDeposit = Transaction::where(['user_id' => $user->id, 'status' => 1, 'type' => 'deposit'])->sum('amount');
        $totalWithdraw = Withdrawal::where(['user_id' => $user->id, 'status' => 1, 'currency' => config('settings')->cur_text])->sum('amount');
        $latestTrx = Transaction::where(['user_id' => $user->id, 'currency' => config('settings')->cur_text])->latest()->limit(5)->get();
        $totalIncome = Transaction::where(['user_id' => $user->id, 'trx_type' => '+', 'currency' => config('settings')->cur_text])
            ->where('type','income')
            ->sum('amount');


        $projectIncome = Transaction::where(['user_id' => $user->id, 'currency' => config('settings')->cur_text, 'trx_type' => '+'])
            ->where(function ($q){
                $q->where('details','LIKE','Profits from%');
            })
            ->sum('amount');


        $referralIncome = Transaction::where(['user_id' => $user->id, 'currency' => config('settings')->cur_text, 'trx_type' => '+'])
            ->where('type','commission')
            ->sum('amount');

        $today = date("Y-m-d 00:00:00");
        $todayIncome = Transaction::where(['user_id' => $user->id, 'currency' => config('settings')->cur_text, 'type' => 'income'])
            ->where('created_at', '>=', $today)
            ->sum('amount');

        $todayBonus = Transaction::where(['user_id' => $user->id, 'currency' => config('settings')->cur_text, 'type' => 'bonus'])
        ->where('created_at', '>=', $today)
        ->sum('amount');

        $totalInvest = Investment::where('user_id', $user->id)->sum('amount');
        $totalBonus = Transaction::where(['type' => 'bonus', 'user_id' => $user->id])->sum('amount');
        $todayIncome = $todayIncome + $todayBonus;
        $totalIncome = $totalIncome + $totalBonus;
		
		$viu = is_null($viewID) ? 'profile':'wallets';

        return view($this->activeTemplate."user.$viu",compact('pageTitle','userBalance','referrals','totalDeposit','totalWithdraw','totalInvest','totalIncome','todayIncome','referralIncome','projectIncome','user'));
    }

    public function view_profile_details(){

        $pageTitle = 'Personal Details';
        return view($this->activeTemplate.'user.profile_details',compact('pageTitle'));
    }


    function sendWithdrawSms(){
        $gnl = GeneralSetting::first();
        $headers = "From: $gnl->sitename <$gnl->email_from> \r\n";
        $headers .= "Reply-To: $gnl->sitename <$gnl->email_from> \r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=utf-8\r\n";

        $code = rand(0,9).rand(0,9).rand(0,9).rand(0,9).rand(0,9).rand(0,9);

        //instant_sms("Your Verification Code is $code",\auth()->user()->mobile);

        session()->put('withdrawal_code', $code);

        @sendGeneralEmail(\auth()->user()->email, "Withdrawal Verification Code", "Hi, your Withdrawal verification code is $code");
        //send_general_email(\auth()->user()->email,"Withdrawal OTP Code","Your Withdrawal verification code is $code");

        //@mail(\auth()->user()->email,"Withdrawal Verification Code","Your Verification Code is $code",$headers);

        return '1';

    }

    public function welcome(){

        // Verify id session set
        //if(empty(session()->get('bonus_welcome'))) return redirect()->route('user.home');

        // Remove Session
        session()->put('bonus_welcome', '');

        $pageTitle = 'Welcome Bonus';
        return view($this->activeTemplate.'user.welcome_bonus',compact('pageTitle'));
    }
	
	
    public function Notification(){
		
		$user = auth()->user();
        $logs = $user->notifications()->where(['read_status' => 0])->orderBy('id','desc')->paginate(getPaginate());

        $pageTitle = 'Users Notifications';
        return view($this->activeTemplate.'user.notice',compact('pageTitle','user','logs'));
    }
	
	
	public function tradeNow( $time, Request $request ){
        $user = User::find(\auth()->user()->id);
		
		$invests = Investment::where('user_id', $user->id)->where('status', 0)->count();

        if (!$invests) {
            $notify[] = ['error', 'You do not have an active investment to intiate trade'];
            return $this->handleResponse($request, 'You do not have an active investment to intiate trade', 401, $notify);
        }
		
		$update = Investment::where('user_id', $user->id)->where('status', 0)->update([
			'is_started' => 0
		]);

        $nextReturn = Carbon::now()->addHours(24);
		$user->next_claim = $nextReturn;
		$user->save();

		$bots = ['Oxcar','Kryll','Gunbot','Shrimpy','Coinrule','Coinigy','3Commas','Pionex','Burat','Exness','Bitsgap','Cryptohopper','WunderTrading','Quadency','Cornix','TradeSanta']; shuffle($bots);
		$botName = $bots[2];

		$notify[] = ['success',"Congratulations! $botName will be trading for you within the next 24 hours"];
        return back()->withNotify($notify);
	}

	
	public function takeReward( $invID, Request $request ){
        $user = User::find(\auth()->user()->id);
		
		$invests = Investment::where('user_id', $user->id)->where('id', $invID)->first();

        if (!$invests) {
            $notify[] = ['error', 'You do not have an active investment to intiate earn'];
            return $this->handleResponse($request, 'You do not have an active investment to intiate earn', 401, $notify);
        }
		
		if( $invests->is_started == 0 ){
            $notify[] = ['error', 'You have already receive earnings for today'];
            return $this->handleResponse($request, 'You have already receive earnings for today', 401, $notify);
		}
		
		$invests->is_started = 0;
		$amount = $invests->interest_amount;

		// NGN Currency
		if($invests->currency == 'NGN') {
			$user->income += $amount;
			$user->save();
			$post_balance = $user->income;
		}

		// USDT Currency
		if($invests->currency == 'USDT') {
			$user->usdt_income += $amount;
			$user->save();
			$post_balance = $user->usdt_income;
		}
		$invests->earned = $invests->earned + $amount;

        // Add Transaction
        $transaction = new Transaction();
        $transaction->user_id = $user->id;
        $transaction->type =  'income';
        $transaction->amount = $amount;
        $transaction->post_balance = $post_balance;
        $transaction->charge = 0;
        $transaction->currency = $invests->currency;
        $transaction->trx_type = '+';
        $transaction->details = 'Income Received';
        $transaction->trx =  getTrx();
        $transaction->status =  1;
        $transaction->save();
		
		if( config('settings')->commission_type == 'rebate' || config('settings')->commission_type == 'both' )
		{
			$creditBonus = teamRebate($user, $invests->currency, $amount, 2);
			// Update Invest
			if($creditBonus instanceof Transaction) {
				$invests->ref_bonus += $creditBonus->amount;
			}
		}

		$invests->save();

        $nextReturn = Carbon::now()->addHours(24);
		$user->next_claim = $nextReturn;
		$user->save();

        return $this->handleResponseRedirect($request, config('settings')->cur_sym . getAmount($post_balance), 200, url(route('user.investment.log')));
	}


}
