<?php
namespace App\Http\Services\Payout\Bank;

use App\Http\Services\Payout\Bank\Util\ResponseBankUtilPayoutServices;
use App\Models\Bank;
use App\Models\Gateway;
use Illuminate\Support\Facades\URL;

/**
 * Laravel/Symfony Developer
 * Name: Pynith Limited
 * Telegram: @pynith
 * Hire me via Telegram: pynith
 */
class OTPayBankPayoutServices
{
    
    /**
     * Transfer Payment
     * 
     * @param string reference
     * @param string currency
     * @param string amount
     * @param string method
     * @param string bank code
     * @param string bank name
     * @param string account number
     * @param string account name
     * @param string narration
     */
    public function transfer(
        string $reference,
        string $currency,
        string $amount,
        string $method,
        string $bank_code,
        string $bank_name = null,
        string $account_number,
        string $account_name,
        string $narration = 'Goods'
    )
    {
        try {

            // Get Support Code
            $bank_list = Bank::where(['bank_code' => $bank_code, 'status' => 1])->first();

            // Exception
            if(!$bank_list) throw new \Exception('Bank Name not available. Please update your bank info again');

			$user = auth()->user();

            // Find Setting
            $setting = Gateway::where(['code' => '104', 'status' => 1])->first();

            // Exception
            if(!$setting) throw new \Exception("Service not enabled at the moment");

            // Parse setting
            $setting = json_decode($setting->gateway_parameters);

            // Request Body
            $requestBody = array(
                "merchantId" => intval($setting->merchantId->value),
                "merchantOrderId" => $reference,
                "amount" 		=> getAmount($amount),
            );
			
            // Build Sign
            $sign = self::buildSignDigest($requestBody, $setting->appSecret->value);
            $requestBody['sign'] = $sign;
			
            // Other Request
            $otRequest = array(
				"currency" 			=> "NGN",
                "fundAccount" => [
					"accountType" => 'bank_account',
					"contact" => [
						"email" => $user->email,
						"mobile" => $user->mobile,
						"name" => $account_name
					],
					"bankAccount" => [
						"accountNumber" => $account_number,
						"bankCode" => $bank_list->bank_code,
						"name" => $account_name
					],
				],
                "notifyUrl" => '' . URL::route('ipn.otpay') . ''
            );
			
			$dataSet = array_merge($requestBody, $otRequest);
			
            // Send Curl Request
            $sendRequest = $this->curlRequest('POST', 'api/payout/submit', $dataSet);
			
            // Exception
            if($sendRequest instanceof \Exception) throw new \Exception($sendRequest['error']);

            // Check if successful
            if ($sendRequest['code'] != 0) throw new \Exception($sendRequest['error']);

            // Format Response
			$response = ResponseBankUtilPayoutServices::response(200, 2, true, $reference, $sendRequest['data']['payoutId'], $currency, $amount, $method, $bank_code, $bank_name, $account_number, $account_name, $narration);

            // Response
            return $response;

        } catch (\Exception $th) {
            //throw $th;
            return $th;
        }
    }

    /**
     * Get Account Balance
     * 
     * @param Void
     */
    public function bank_list() {
        try {

            // Request Body
            $requestBody = array(
                "merchantId" => '****',
                "timestamp" => time(),
            );

            // Send Curl Request
            $sendRequest = $this->curlRequest('POST', 'api/payout/bankList', $requestBody);

            // Exception
            if ($sendRequest instanceof \Exception) throw new \Exception($sendRequest->getMessage());

            // Response
            return $sendRequest;
        } catch (\Exception $th) {
            return $th;
        }
    }


    /**
     * Cur Request
     * send request
     * 
     * @param string method
     * @param string endpoint
     * @param string body
     */
    public function curlRequest(
        string $method,
        string $endpoint,
        array $body = []
    ) {

        try {

            // Convert request data to JSON
            $jsonData = json_encode($body);

            // Curl Request
            $curl = curl_init();
            curl_setopt_array($curl, array(
                CURLOPT_URL => 'https://pay.otpay.io/' . $endpoint,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 60,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => $method,
                CURLOPT_POSTFIELDS => $jsonData,
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
				    'Content-Length: ' . strlen($jsonData)
                ),
            ));

            $response = curl_exec($curl);
            $err = curl_error($curl);
            $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

            // there was an error contacting
            if ($err) throw new \Exception('Curl returned error: ' . $err);

            // Convert Json to Json Array
            $response_array = json_decode($response, true);
			
            // Check status
            if (!in_array($httpcode, ['200'])) throw new \Exception($response_array['error']);

            return $response_array;
        } catch (\Exception $th) {
            //throw $th;
            return $th;
        }
    }

    public static function buildSignDigest($data, $secret_key)
    {

        // Concatenate the string in the format key=value&key=value
        $signString = '';
        foreach ($data as $key => $value) {
            if (!empty($value)) {
                $signString .= $key . '=' . $value . '&';
            }
        }

        // Append the private key to the string
        $signString .= 'appSecret=' . $secret_key;

        // Perform MD5 signature on the generated string
        $sign = md5($signString);
		
        return $sign;

    }
}
